#!/bin/bash
set -e

echo "=== Starting filesystem expansion ==="

ROOT_PART="$(findmnt / -o source -n)"
ROOT_DEV="/dev/$(lsblk -no pkname "$ROOT_PART")"
PART_NUM="$(echo "$ROOT_PART" | grep -o "[[:digit:]]*$")"

echo "Root partition: $ROOT_PART"
echo "Root device: $ROOT_DEV"
echo "Partition number: $PART_NUM"

# Safety check: verify this is the last partition
LAST_PART_NUM=$(parted "$ROOT_DEV" -ms unit s p 2>/dev/null | tail -n 1 | cut -f 1 -d:)
if [ "$LAST_PART_NUM" != "$PART_NUM" ]; then
  echo "ERROR: $ROOT_PART (partition $PART_NUM) is not the last partition (last is $LAST_PART_NUM)"
  echo "Cannot safely expand - manual intervention required"
  exit 1
fi

# Check if growpart is available
if ! command -v growpart &> /dev/null; then
    echo "ERROR: growpart not found. Install: pacman -S cloud-guest-utils"
    exit 1
fi

# Expand partition
echo "Expanding partition $PART_NUM on $ROOT_DEV..."
if growpart "$ROOT_DEV" "$PART_NUM"; then
    echo "Partition expanded successfully"
elif [ $? -eq 1 ]; then
    echo "Partition already at maximum size"
    exit 0
else
    echo "ERROR: growpart failed with code $?"
    exit 1
fi

# Give kernel time to update partition table
sleep 2
partprobe "$ROOT_DEV" || true

# Resize filesystem
echo "Resizing filesystem on $ROOT_PART..."
resize2fs "$ROOT_PART"

echo "=== Filesystem expansion complete ==="
