#!/bin/bash

# Source common reset functions
source /etc/stellarmate/reset-factory-common.sh

# Fix file ownerships
function fix_ownerships() {
    echo "Setting correct file ownerships..."
    
    # Only fix specific directories we modified during reset
    echo "  - Setting /home/stellarmate ownership..."
    chown stellarmate:stellarmate /home/stellarmate
    
    echo "  - Setting .config ownership..."
    chown -R stellarmate:stellarmate /home/stellarmate/.config 2>/dev/null || true
    
    # .local subdirectories - EXCLUDING flatpak
    echo "  - Setting .local subdirectories (excluding flatpak)..."
    chown -R stellarmate:stellarmate /home/stellarmate/.local/share/kstars 2>/dev/null || true
    chown -R stellarmate:stellarmate /home/stellarmate/.local/share/ekoslive 2>/dev/null || true
    chown -R stellarmate:stellarmate /home/stellarmate/.local/share/rustdesk-reset 2>/dev/null || true
    # Skip flatpak - it's huge and has special mount points
    # chown -R stellarmate:stellarmate /home/stellarmate/.local/share/flatpak
    
    echo "  - Setting .var ownership..."
    chown -R stellarmate:stellarmate /home/stellarmate/.var 2>/dev/null || true
    
    echo "  - Setting .indi ownership..."
    chown -R stellarmate:stellarmate /home/stellarmate/.indi 2>/dev/null || true
    
    echo "  - Setting user directories ownership..."
    chown -R stellarmate:stellarmate /home/stellarmate/Pictures 2>/dev/null || true
    chown -R stellarmate:stellarmate /home/stellarmate/Documents 2>/dev/null || true
    chown -R stellarmate:stellarmate /home/stellarmate/Downloads 2>/dev/null || true
    
    echo "  - Setting NetworkManager ownership..."
    chown -R root:root /etc/NetworkManager
    
    echo "Ownership fix complete"
}

# Reset KStars Flatpak configuration
function reset_kstars_flatpak() {
    echo "Resetting KStars Flatpak configuration..."
    
    # Copy KStars config files
    cp /etc/stellarmate/kstars/config/* /home/stellarmate/.var/app/org.kde.kstars/config/
    
    # Reset KStars data directory
    rm -rf /home/stellarmate/.var/app/org.kde.kstars/data/kstars
    mkdir -p /home/stellarmate/.var/app/org.kde.kstars/data/kstars
    
    # Copy catalog files
    cp /etc/stellarmate/kstars/catalogs/* /home/stellarmate/.var/app/org.kde.kstars/data/kstars/
}

# Reset Plasma desktop settings
function reset_plasma_settings() {
    echo "Resetting Plasma desktop settings..."
    cp -rf /etc/stellarmate/plasma/* /home/stellarmate/.config
}

# Reset boot partition to defaults
function reset_boot_partition() {
    echo "Resetting boot to default..."
    
    # Copy default boot files
    cp /etc/stellarmate/boot/* /boot/

    # Update cmdline.txt with current system version for BTRFS subvolume
    if [ -f /etc/stellarmate/version ] && [ -f /boot/cmdline.txt ]; then
        VERSION=$(cat /etc/stellarmate/version)
        echo "Updating /boot/cmdline.txt with version $VERSION"
        sed -i "s/system-2.0.0/system-$VERSION/g" /boot/cmdline.txt
    fi
    
    # Clean up Raspberry Pi specific files
    rm -f /boot/.firmware_revision || true
    rm -rf /boot.bak || true
}

# Enable filesystem resize on next boot
function enable_filesystem_resize() {
    echo "Enabling resize file system service..."
    systemctl enable resize-fs.service
}

# Perform SM-130 specific operations
function perform_sm130_reset() {
    echo "====================================="
    echo "Starting SM-130 Specific Reset"
    echo "====================================="
    
    # KStars and Plasma
    reset_kstars_flatpak
    reset_plasma_settings
    
    # Boot configuration
    reset_boot_partition
    
    # Filesystem resize
    enable_filesystem_resize
    
    # Final ownership fix (needed after all file operations)
    echo "Setting final permissions..."
    fix_ownerships
    
    echo "====================================="
    echo "SM-130 Reset Complete"
    echo "====================================="
}

# Main execution
function main() {
    echo "========================================"
    echo "StellarMate SM-130 Factory Reset"
    echo "========================================"
    
    # Perform common reset operations
    perform_common_reset "stellarmate"
    
    # Perform SM-130 specific operations
    perform_sm130_reset
    
    echo "========================================"
    echo "Reset complete."
    echo "StellarMate must be restarted now."
    echo "========================================"
}

# Run main function
main
